#ifndef __SCalCLI__
#define __SCalCLI__

/**
 * @file SCalMain.cpp
 * @brief Main SCal function. Entry point to application
 * $Id: SCalMain.cpp,v 1.1 2006/10/27 07:45:08 paul Exp $
 */

#include <ECore.h>
#include <EMusic.h>

/**
 * @fn void ShowHelp()
 * @brief Show the help infomation
 */
void ShowHelp()
{
	cout << "+----------------------------------------------------+" << endl;
	cout << "| Exponent SCal audio computer                       |" << endl;
	cout << "| Copyright 2006 Exponent Digital                    |" << endl;
	cout << "| Visit us online: http://www.expdigital.co.uk       |" << endl;
	cout << "+----------------------------------------------------+" << endl;
	cout << "| Usage:                                             |" << endl;
	cout << "| SCal [calculator] {parameters}                     |" << endl;
	cout << "|                                                    |" << endl;
	cout << "| Types of calculator:                               |" << endl;
	cout << "| -d Delay time                                      |" << endl;
	cout << "| -l Lfo time                                        |" << endl;
	cout << "| -r Closest rate                                    |" << endl;
	cout << "| -n Closest note                                    |" << endl;
	cout << "| -f Note frequency                                  |" << endl;
	cout << "| -s Tempo Note Shift                                |" << endl;
	cout << "| -t Timestretch                                     |" << endl;
	cout << "| -h Tempo shift                                     |" << endl;
	cout << "| -b Bpm from time                                   |" << endl;
	cout << "| -i Time from Bpm                                   |" << endl;
	cout << "|                                                    |" << endl;
	cout << "| Use SCal [calculator] -help for parameter options  |" << endl;
	cout << "+----------------------------------------------------+" << endl;

}

/**
 * @fn void ShowDelayHelp()
 * @brief Show the help infomation
 */
void ShowDelayHelp()
{
	cout << "+----------------------------------------------------+" << endl;
	cout << "| Computes the delay time in seconds                 |" << endl;
	cout << "| -d Bpm Division TimeSignature                      |" << endl;
	cout << "| Bpm is the input BPM                               |" << endl;
	cout << "| Division is the tempo divison eg 1/8               |" << endl;
	cout << "|     - To make triplet add a T at the end - 1/8T    |" << endl;
	cout << "|     - To make dotted add a D at the end  - 1/8D    |" << endl;
	cout << "| TimeSignature is the time signature eg 4/4         |" << endl;
	cout << "+----------------------------------------------------+" << endl;
}

/**
 * @fn void ShowLfoHelp()
 * @brief Show the help information for the lfo
 */
void ShowLfoHelp()
{
	cout << "+----------------------------------------------------+" << endl;
	cout << "| Computes the lfo rate in hertz                     |" << endl;
	cout << "| -d Bpm Division TimeSignature                      |" << endl;
	cout << "| Bpm is the input BPM                               |" << endl;
	cout << "| Division is the tempo divison eg 1/8               |" << endl;
	cout << "|     - To make triplet add a T at the end - 1/8T    |" << endl;
	cout << "|     - To make dotted add a D at the end  - 1/8D    |" << endl;
	cout << "+----------------------------------------------------+" << endl;
}

/**
 * Convert a string to a division
 * @param div The division as a string
 * @retval CMusicMath::ETempoDivision The division
 */
CMusicMath::ETempoDivision ConvertStringToDivision(const CString &div)
{
	return (CMusicMath::ETempoDivision)0;
}

/**
 * Convert to a time signature
 * @param ts The time sig as a string
 * @retval CTimeSignature The time signature
 */
CTimeSignature ConvertStringToTimeSignature(const CString &ts)
{
	CStringTokeniser tokeniser(ts, '/');

	// Check if we have enough token
	if (tokeniser.getNumberOfTokens != 2)
	{
		return CTimeSignature(4, 4);
	}

	CString numerator;
	CString denominator;

	// Get the two tokens
	if (!(tokeniser.getNextToken(numerator) && tokeniser.getNextToken(denominator)))
	{
		return CTimeSignature(4, 4);
	}

	return CTimeSignature(atol(numerator.getString()), atol(denominator.getString()));
}

/**
 * @fn int main(int argc, char *argv[])
 * @param argc The argument count
 * @param argv The arguments
 * @retval int Returns 0
 */
int main(int argc, char *argv[])
{
	// Okay so we are gonna open a window!
#ifdef WIN32
	START_EXPONENT_API(NULL, "Exponent", "SCal V5", "Muscial value calculator", "5.0", "Software//Exponent//SCal", "2006 Exponent Digital");
#else
	START_EXPONENT_API("Exponent", "SCal V5", "Muscial value calculator", "5.0", "Software//Exponent//SCal", "2006 Exponent Digital");
#endif

	// Check if we actually have the proper number of arguments
	if (argc < 3)
	{
		ShowHelp();
		return 0;
	}

	switch(argv[1][1])
	{
		case 'd':
			// Delay parameters
			if (strcmp(argv[2], "-help") == 0)
			{
				ShowDelayHelp();
				return 0;
			}
			else
			{
				// Check we have the required parameters
				if (argc != 5)
				{
					ShowDelayHelp();
					return 0;
				}

				// Convert all the parameters
				const double bpm 	  = atof(argv[2]);
				CString division 	  = argv[3];
				CString timeSignature = argv[4];

				// Output
				cout << "Delay time in msec = " << CMusicMath::getRateInMilliseconds(bpm, ConvertStringToDivision(division), ConvertStringToTimeSignature(timeSignature)) << endl;
			}
		break;
		case 'l':
			// Lfo parameters
			if (strcmp(argv[2], "-help") == 0)
			{
				ShowLfoHelp();
				return 0;
			}
			else
			{
				// Check we have the required parameters
				if (argc != 4)
				{
					ShowLfoHelp();
					return 0;
				}

				// Convert all the parameters
				const double bpm 	  = atof(argv[2]);
				CString division 	  = argv[3];

				// Output
				cout << "LFO rate in Hz = " << CMusicMath::getRateInHertz(bpm, ConvertStringToDivision(division)) << endl;
			}
		break;
		case 'r':
		/*
			if (strcmp(argv[2], "-help") == 0)
			{
				ShowLfoHelp();
				return 0;
			}
			else
			{

			}
		*/
		break;
		case 'n':
		/*
			if (strcmp(argv[2], "-help") == 0)
			{
				ShowLfoHelp();
				return 0;
			}
			else
			{

			}
		*/
		break;
		case 'f':
		/*
			if (strcmp(argv[2], "-help") == 0)
			{
				ShowLfoHelp();
				return 0;
			}
			else
			{

			}
		*/
		break;
		case 's':
		/*
			if (strcmp(argv[2], "-help") == 0)
			{
				ShowLfoHelp();
				return 0;
			}
			else
			{

			}
		*/
		break;
		case 't':
		/*
			if (strcmp(argv[2], "-help") == 0)
			{
				ShowLfoHelp();
				return 0;
			}
			else
			{

			}
		*/
		break;
		case 'h':
		/*
			if (strcmp(argv[2], "-help") == 0)
			{
				ShowLfoHelp();
				return 0;
			}
			else
			{

			}
		*/
		break;
		case 'b':
		/*
			if (strcmp(argv[2], "-help") == 0)
			{
				ShowLfoHelp();
				return 0;
			}
			else
			{

			}
		*/
		break;
		case 'i':
		/*
			if (strcmp(argv[2], "-help") == 0)
			{
				ShowLfoHelp();
				return 0;
			}
			else
			{

			}
		*/
		break;
	}

	return 0;
}

#endif		// End of SCal.cpp